/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/

/**
 * @file rtCompare.h  Functions for comparing the 
 *                    values of primitive ASN.1 types.
 */

#ifndef _RTCOMPARE_H_
#define _RTCOMPARE_H_

#if !defined NO_ASN1_RT_REDUNDANT
#   define NO_ASN1_RT_REDUNDANT
#endif /* !NO_ASN1_RT_REDUNDANT */

#include "asn1type.h"

#ifdef __cplusplus
extern "C" {
#endif

/** 
 * @defgroup cmp Comparison Functions 
 * @ingroup cruntime 
 * @{
 *
 * The group of functions allows comparing the values of primitive ASN.1 types.
 * These functions are used in the comparison routines that are generated by
 * the ASN1C complier when the <i>-gencompare</i> option is used.
 *
 * Information on elements that do not match is written to the given error
 * buffer for each function. This makes it possible to compare complex
 * structures and get back specific information as to what elements within
 * those structures are different.
 */
/* Compare function to Buffer */

/**
 * The rtCmpBoolean function compares two ASN.1 Boolean values. The return
 * value is TRUE (matched) or FALSE (unmatched).
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpBoolean (ASN1ConstCharPtr name, ASN1BOOL value, 
                                ASN1BOOL compValue, char* errBuff, 
                                int errBuffSize);

EXTERNRT ASN1BOOL rtCmpInt8 (ASN1ConstCharPtr name, ASN1INT8 value, 
                             ASN1INT8 compValue, char* errBuff, 
                             int errBuffSize);

EXTERNRT ASN1BOOL rtCmpSInt (ASN1ConstCharPtr name, ASN1SINT value, 
                             ASN1SINT compValue, char* errBuff, 
                             int errBuffSize);

EXTERNRT ASN1BOOL rtCmpUInt8 (ASN1ConstCharPtr name, ASN1UINT8 value, 
                              ASN1UINT8 compValue, char* errBuff, 
                              int errBuffSize);

EXTERNRT ASN1BOOL rtCmpUSInt (ASN1ConstCharPtr name, ASN1USINT value, 
                              ASN1USINT compValue, char* errBuff, 
                              int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpInteger function compars two ASN.1 INTEGER values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpInteger (ASN1ConstCharPtr name, ASN1INT value, 
                                ASN1INT compValue, char* errBuff, 
                                int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpUnsigned function compares two ASN.1 unsigned INTEGER values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpUnsigned (ASN1ConstCharPtr name, ASN1UINT value, 
                                 ASN1UINT compValue, char* errBuff, 
                                 int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCompInt64 function compares two 64-bit ASN.1 INTEGER values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpInt64 (ASN1ConstCharPtr name, ASN1INT64 value, 
                              ASN1INT64 compValue, char* errBuff, 
                              int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpUInt64 function compares two 64-bit ASN.1 unsigned INTEGER values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpUInt64 (ASN1ConstCharPtr name, ASN1UINT64 value, 
                               ASN1UINT64 compValue, char* errBuff, 
                               int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpBitStr function compares two ASN.1 BIT STRING values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param numbits      The number of bits in the first value to compare.
 * @param data         The pointer to the data of the first value to compare.
 * @param compNumbits  The number of bits in the second value to compare.
 * @param compData     The pointer to the data of the second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpBitStr (ASN1ConstCharPtr name, ASN1UINT numbits, 
                               ASN1ConstOctetPtr data, ASN1UINT compNumbits, 
                               ASN1ConstOctetPtr compData, char* errBuff, 
                               int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpOctStr function compares two ASN.1 OCTET STRING values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param numocts      The number of the octets in the first value to compare.
 * @param data         The pointer to the data of the first value to compare.
 * @param compNumocts  The number of the octets in the second value to compare.
 * @param compData     The pointer to the data of the second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpOctStr (ASN1ConstCharPtr name, ASN1UINT numocts, 
                               ASN1ConstOctetPtr data, ASN1UINT compNumocts, 
                               ASN1ConstOctetPtr compData, char* errBuff, 
                               int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpCharStr function compares two ASN.1 8-bit character sting values
 * (including IA5String, VisibleString, PrintableString, NumericString, etc.)
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param cstring      The first standard null-terminated string to compare.
 * @param compCstring  The second standard null-terminated string to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpCharStr (ASN1ConstCharPtr name, ASN1ConstCharPtr cstring, 
                                ASN1ConstCharPtr compCstring, char* errBuff, 
                                int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmp16BitCharStr function compares two ASN.1 16-bit character string
 * values (including BMPString).
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param bstring      The pointer to the first 16-bit character string
 *                       structure to compare.
 * @param compBstring  The pointer to the second 16-bit character string
 *                       structure to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmp16BitCharStr (ASN1ConstCharPtr name, 
                                     Asn116BitCharString* bstring, 
                                     Asn116BitCharString* compBstring, 
                                     char* errBuff, int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmp32BitCharStr function compares two 32-bit character string values
 * (including UniversalString).
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param bstring      The pointer to the first 32-bit character string
 *                       structure to compare.
 * @param compBstring  The pointer to the second 32-bit character string
 *                       structure to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmp32BitCharStr (ASN1ConstCharPtr name, 
                                     Asn132BitCharString* bstring, 
                                     Asn132BitCharString* compBstring, 
                                     char* errBuff, int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpReal function compares two ASN.1 REAL values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param value        First value to compare.
 * @param compValue    Second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpReal (ASN1ConstCharPtr name, ASN1REAL value, 
                             ASN1REAL compValue, char* errBuff, 
                             int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpOID function compares two ASN.1 OBJECT IDENTIFIER or REALTIVE-OID
 * values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param pOID         The pointer to the first value to compare.
 * @param pcompOID     The pointer to the second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
  /* dim: Standart_CC only */
EXTERNRT ASN1BOOL rtCmpOID (ASN1ConstCharPtr name, ASN1OBJID* pOID, 
                            ASN1OBJID* pcompOID, char* errBuff, 
                            int errBuffSize);

EXTERNRT ASN1BOOL rtCmpOIDValue (ASN1ConstCharPtr name, ASN1OBJID* pOID, 
                                 ASN1OBJID* pcompOID, char* errBuff, 
                                 int errBuffSize);

/**
 * The rtCmpOID64 function compares two 64-bit ASN.1 OBJECT IDENTIFIER or
 * RELATIVE-OID values.
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param pOID         The pointer to the first value to compare.
 * @param pcompOID     The pointer to the second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpOID64 (ASN1ConstCharPtr name, ASN1OID64* pOID, 
                              ASN1OID64* pcompOID, char* errBuff, 
                              int errBuffSize);
EXTERNRT ASN1BOOL rtCmpOID64Value (ASN1ConstCharPtr name, ASN1OID64* pOID, 
                                   ASN1OID64* pcompOID, char* errBuff, 
                                   int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/** 
 * The rtCmpOpenType function compares two ASN.1 values of the old (pre- 1994)
 * ASN.1 ANY type or other elements defined in the later standards to be Open
 * Types (for example, a variable type declaration in a CLASS construct defined
 * in X.681).
 *
 * @param name         The name of value. Used for constructing errBuff if
 *                       values are not matching.
 * @param numocts      The number of octets in the first value to compare.
 * @param data         The pointer to the data of the first value to compare.
 * @param compNumocts  The number of octets in the second value to compare.
 * @param compData     The pointer to the data of the second value to compare.
 * @param errBuff      The pointer to the buffer to receive the null-terminated
 *                       string if the comparison fails. If the comparison
 *                       failed, this buffer will contain the null-terminated
 *                       string that explains the reason of comparison failure.
 * @param errBuffSize  The size of the errBuff buffer.
 * @return             The comparison result. TRUE, if values matched,
 *                       otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpOpenType (ASN1ConstCharPtr name, 
                                 ASN1UINT numocts, ASN1ConstOctetPtr data, 
                                 ASN1UINT compNumocts, ASN1ConstOctetPtr compData, 
                                 char* errBuff, int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * The rtCmpOpenTypeExt function compares two ASN.1 open type extension values.
 *
 * An open type extension is defined as an extensibility marker on a
 * constructed type without any extension elements defined (for example,
 * SEQUENCE {a INTEGER, ...}). The difference is that this is an implicit field
 * that can span one or more elements whereas the standard Open Type is assumed
 * to be a single tagged field.
 *
 * @param name             The name of value. Used for constructing errBuff if
 *                           values are not matching.
 * @param pElemList        The first pointer to a linked list structure. The
 *                           list should consist of ASN1OpenType elements.
 * @param pCompElemList    The second pointer to a linked list structure. The
 *                           list should consist of ASN1OpenType elements.
 * @param errBuff          The pointer to the buffer to receive the
 *                           null-terminated string if the comparison fails. If
 *                           the comparison failed, this buffer will contain
 *                           the null-terminated string that explains the
 *                           reason of comparison failure.
 * @param errBuffSize      The size of the errBuff buffer.
 * @return                 The comparison result. TRUE, if values matched,
 *                           otherwise FALSE.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpOpenTypeExt (ASN1ConstCharPtr name, 
                                    Asn1RTDList* pElemList, 
                                    Asn1RTDList* pCompElemList, 
                                    char* errBuff, int errBuffSize);
EXTERNRT ASN1BOOL rtCmpTag (ASN1ConstCharPtr name, int tag, int compTag, 
                            char* errBuff, int errBuffSize);
EXTERNRT ASN1BOOL rtCmpSeqOfElements (ASN1ConstCharPtr name, int noOfElems, 
                                      int compNoOfElems, char* errBuff, 
                                      int errBuffSize);
EXTERNRT ASN1BOOL rtCmpOptional (ASN1ConstCharPtr name, unsigned presentBit, 
                                 unsigned compPresentBit, char* errBuff, 
                                 int errBuffSize);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @} cmp
 */

/* Compare function To StdOut */

/**@defgroup cmpStdout Comparison to Standard Output Functions
 * @ingroup cruntime 
 * @{ 
 * The rtCmpToStdout<type> functions do the same actions as the rtCmp<type>
 * ones, but they print the comparison results to stdout instead of putting it
 * into the buffer. The prototypes of these functions are almost the same as
 * for the rtCmp<type> except the last two parameters - they re absent in the
 * rtCmpToStdout<type> functions.
 *

 */
/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutBoolean (ASN1ConstCharPtr name, ASN1BOOL value, 
                                        ASN1BOOL compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutInteger (ASN1ConstCharPtr name, ASN1INT value, 
                                        ASN1INT compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutInt64 (ASN1ConstCharPtr name, ASN1INT64 value, 
                                      ASN1INT64 compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutUnsigned (ASN1ConstCharPtr name, ASN1UINT value, 
                                         ASN1UINT compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutUInt64 (ASN1ConstCharPtr name, ASN1UINT64 value, 
                                       ASN1UINT64 compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * @param name         The name of value.
 * @param numbits      The first value to compare.
 * @param data         The pointer to the first value.
 * @param compNumbits  The second value to compare.
 * @param compData     The pointer to the second value.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutBitStr (ASN1ConstCharPtr name, ASN1UINT numbits, 
                                       ASN1ConstOctetPtr data, 
                                       ASN1UINT compNumbits, 
                                       ASN1ConstOctetPtr compData);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param numocts      The first value to compare.
 * @param data         The pointer to the data of the first value.
 * @param compNumocts  The second value to compare.
 * @param compData     The pointer to the data of the second value.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOctStr (ASN1ConstCharPtr name, ASN1UINT numocts, 
                                       ASN1ConstOctetPtr data, 
                                       ASN1UINT compNumocts, 
                                       ASN1ConstOctetPtr compData);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param cstring      The first value to compare.
 * @param compCstring  The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutCharStr (ASN1ConstCharPtr name, 
                                        ASN1ConstCharPtr cstring, 
                                        ASN1ConstCharPtr compCstring);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param bstring      The first value to compare.
 * @param compBstring  The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdout16BitCharStr (ASN1ConstCharPtr name, 
                                             Asn116BitCharString* bstring, 
                                             Asn116BitCharString* compBstring);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param bstring      The first value to compare.
 * @param compBstring  The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdout32BitCharStr (ASN1ConstCharPtr name, 
                                             Asn132BitCharString* bstring, 
                                             Asn132BitCharString* compBstring);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param value        The first value to compare.
 * @param compValue    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutReal (ASN1ConstCharPtr name, 
                                     ASN1REAL value, ASN1REAL compValue);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param pOID         The first value to compare.
 * @param pcompOID     The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOID (ASN1ConstCharPtr name, 
                                    ASN1OBJID* pOID, ASN1OBJID* pcompOID);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param pOID         The first value to compare.
 * @param pcompOID     The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOIDValue (ASN1ConstCharPtr name, 
                                         ASN1OBJID* pOID, ASN1OBJID* pcompOID);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param pOID         The first value to compare.
 * @param pcompOID     The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOID64 (ASN1ConstCharPtr name, 
                                      ASN1OID64* pOID, ASN1OID64* pcompOID);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param pOID         The first value to compare.
 * @param pcompOID     The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOID64Value (ASN1ConstCharPtr name, 
                                           ASN1OID64* pOID, 
                                           ASN1OID64* pcompOID);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name         The name of value.
 * @param numocts      The number of octets in the first value to compare.
 * @param data         The pointer to the data in the frist value to compare.
 * @param compNumocts  The number of octets in the second value to compare.
 * @param compData     The pointer to the data in the second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOpenType (ASN1ConstCharPtr name, 
                                         ASN1UINT numocts, 
                                         ASN1ConstOctetPtr data, 
                                         ASN1UINT compNumocts, 
                                         ASN1ConstOctetPtr compData);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name		The name of value.
 * @param pElemList 	The first value to compare.
 * @param pCompElemList	The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOpenTypeExt (ASN1ConstCharPtr name, 
                                            Asn1RTDList* pElemList, 
                                            Asn1RTDList* pCompElemList);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name		The name of value.
 * @param tag		The first value to compare.
 * @param compTag	The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutTag (ASN1ConstCharPtr name, int tag, 
                                    int compTag);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name             The name of value.
 * @param noOfElems        The first value to compare.
 * @param compNoOfElems    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutSeqOfElements (ASN1ConstCharPtr name, 
                                              int noOfElems, 
                                              int compNoOfElems);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @param name              The name of value.
 * @param presentBit        The first value to compare.
 * @param compPresentBit    The second value to compare.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT ASN1BOOL rtCmpToStdoutOptional (ASN1ConstCharPtr name, 
                                         unsigned presentBit, 
                                         unsigned compPresentBit);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/**
 * @} cmpStdout
 */


#ifdef __cplusplus
}
#endif
#endif 

